/*____________________________________________________________________________
	Copyright (C) 2000 Networks Associates Technology, Inc.
	All rights reserved.

	$Id: pgpLDAP.c,v 1.41 2001/03/30 19:34:52 dallen Exp $
____________________________________________________________________________*/
#include <string.h>

#include "pgpLDAPPriv.h"
#include "pgpBERPriv.h"
#include "pgpPFLPriv.h"
#include "pgpErrors.h"
#include "pgpMem.h"
#include "pgpSockets.h"
#include "pgpUtilities.h"
#include "pgpContext.h"
#include "pgpStrings.h"


typedef struct PGPldapResultEntry
{
	PGPldapResult	result;
	PGPError		err;
} PGPldapResultEntry;

#define EN(number, err)		{ number, err }

static const PGPldapResultEntry sResults[] =
{
EN( kPGPldapResult_Success,					kPGPError_NoErr ),
EN( kPGPldapResult_OperationsError,			kPGPError_LDAPOperationsError ),
EN( kPGPldapResult_ProtocolError,			kPGPError_LDAPProtocolError ),
EN( kPGPldapResult_TimelimitExceeded,		kPGPError_LDAPTimelimitExceeded ),
EN( kPGPldapResult_SizelimitExceeded,		kPGPError_LDAPSizelimitExceeded ),
EN( kPGPldapResult_CompareFalse, 			kPGPError_NoErr ),
EN( kPGPldapResult_CompareTrue,				kPGPError_NoErr ),
EN( kPGPldapResult_StrongAuthNotSupported,	kPGPError_LDAPStrongAuthNotSupported ),
EN( kPGPldapResult_StrongAuthRequired,		kPGPError_LDAPStrongAuthRequired ),
EN( kPGPldapResult_PartialResults,			kPGPError_LDAPPartialResults ),
EN( kPGPldapResult_NoSuchAttribute,			kPGPError_LDAPNoSuchAttribute ),
EN( kPGPldapResult_UndefinedType,			kPGPError_LDAPUndefinedType ),
EN( kPGPldapResult_InappropriateMatching,	kPGPError_LDAPInappropriateMatching ),
EN( kPGPldapResult_ConstraintViolation,		kPGPError_LDAPConstraintViolation ),
EN( kPGPldapResult_TypeOrValueExists,		kPGPError_LDAPTypeOrValueExists ),
EN( kPGPldapResult_InvalidSyntax,			kPGPError_LDAPInvalidSyntax ),
EN( kPGPldapResult_NoSuchObject, 			kPGPError_LDAPNoSuchObject ),
EN( kPGPldapResult_AliasProblem, 			kPGPError_LDAPAliasProblem ),
EN( kPGPldapResult_InvalidDNSyntax,			kPGPError_LDAPInvalidDNSyntax ),
EN( kPGPldapResult_IsLeaf,					kPGPError_LDAPIsLeaf ),
EN( kPGPldapResult_AliasDerefProblem,		kPGPError_LDAPAliasDerefProblem ),
EN( kPGPldapResult_InappropriateAuth,		kPGPError_LDAPInappropriateAuth ),
EN( kPGPldapResult_InvalidCredentials,		kPGPError_LDAPInvalidCredentials ),
EN( kPGPldapResult_InsufficientAccess,		kPGPError_LDAPInsufficientAccess ),
EN( kPGPldapResult_Busy,					kPGPError_LDAPBusy ),
EN( kPGPldapResult_Unavailable,				kPGPError_LDAPUnavailable ),
EN( kPGPldapResult_UnwillingToPerform,		kPGPError_LDAPUnwillingToPerform ),
EN( kPGPldapResult_LoopDetect,				kPGPError_LDAPLoopDetect ),
EN( kPGPldapResult_NamingViolation,			kPGPError_LDAPNamingViolation ),
EN( kPGPldapResult_ObjectClassViolation, 	kPGPError_LDAPObjectClassViolation ),
EN( kPGPldapResult_NotAllowedOnNonleaf,		kPGPError_LDAPNotAllowedOnNonleaf ),
EN( kPGPldapResult_NotAllowedOnRDN,			kPGPError_LDAPNotAllowedOnRDN ),
EN( kPGPldapResult_AlreadyExists,			kPGPError_LDAPAlreadyExists ),
EN( kPGPldapResult_NoObjectClassMods,		kPGPError_LDAPNoObjectClassMods ),
EN( kPGPldapResult_ResultsTooLarge,			kPGPError_LDAPResultsTooLarge ),
EN( kPGPldapResult_Other,					kPGPError_LDAPOther ),
EN( kPGPldapResult_ServerDown,				kPGPError_LDAPServerDown ),
EN( kPGPldapResult_LocalError,				kPGPError_LDAPLocalError ),
EN( kPGPldapResult_EncodingError,			kPGPError_LDAPEncodingError ),
EN( kPGPldapResult_DecodingError,			kPGPError_LDAPDecodingError ),
EN( kPGPldapResult_Timeout,					kPGPError_LDAPTimeout ),
EN( kPGPldapResult_AuthUnknown,				kPGPError_LDAPAuthUnknown ),
EN( kPGPldapResult_FilterError,				kPGPError_LDAPFilterError ),
EN( kPGPldapResult_UserCancelled,			kPGPError_LDAPUserCancelled ),
EN( kPGPldapResult_ParamError,				kPGPError_LDAPParamError ),
EN( kPGPldapResult_NoMemory,				kPGPError_OutOfMemory ),
EN( kPGPldapResult_ConnectError, 			kPGPError_LDAPConnectError )
};

#define kPGPldapResult_NumResultTableEntries		\
	( sizeof( sResults ) / sizeof( sResults[ 0 ] ) )

#undef EN


	PGPError
PGPNewLDAPContext(
	PGPContextRef		context,
	PGPldapContextRef *	pgpLDAP )

{
    PGPError        err     = kPGPError_NoErr;
	PGPMemoryMgrRef	memMgr	= kInvalidPGPMemoryMgrRef;

    PGPValidatePtr( pgpLDAP );
	PGPValidateContext( context );

	memMgr = PGPPeekContextMemoryMgr( context );

    /* Allocate the PGPldap struct */
    *pgpLDAP =	PGPNewData( memMgr,
					sizeof( PGPldapContext ),
					kPGPMemoryMgrFlags_Clear );
	if( IsNull( pgpLDAP ) )
        return kPGPError_OutOfMemory;

    (*pgpLDAP)->hostname		= NULL;
    (*pgpLDAP)->port			= 0;
    (*pgpLDAP)->sock			= kInvalidPGPSocketRef;
	(*pgpLDAP)->nextMessageID	= 0;
	(*pgpLDAP)->response		= NULL;
    (*pgpLDAP)->sizelimit		= 0;
    (*pgpLDAP)->timelimit		= 0;
    (*pgpLDAP)->deref			= kPGPldapDeref_Never;
    (*pgpLDAP)->result			= kPGPldapResult_Success;
    (*pgpLDAP)->matched			= NULL;
    (*pgpLDAP)->message			= NULL;

    (*pgpLDAP)->threadFns.pgpLDAPThread_MutexAlloc     = NULL;
    (*pgpLDAP)->threadFns.pgpLDAPThread_MutexFree      = NULL;
    (*pgpLDAP)->threadFns.pgpLDAPThread_MutexLock      = NULL;
    (*pgpLDAP)->threadFns.pgpLDAPThread_MutexUnlock    = NULL;
    (*pgpLDAP)->threadFns.pgpLDAPThread_SetErrno       = NULL;
    (*pgpLDAP)->threadFns.pgpLDAPThread_GetErrno       = NULL;
    (*pgpLDAP)->threadFns.pgpLDAPThread_SetLDAPErrno   = NULL;
    (*pgpLDAP)->threadFns.pgpLDAPThread_GetLDAPErrno   = NULL;

    (*pgpLDAP)->memMgr = memMgr;

	return err;
}


	PGPError
PGPNewLDAPMessage(
	PGPldapContextRef	pgpLDAP,
	PGPldapMessageRef *	message )
{
	PGPError		err		= kPGPError_NoErr;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidatePtr( message );

	*message =	PGPNewData( pgpLDAP->memMgr,
					sizeof( PGPldapMessage ),
					kPGPMemoryMgrFlags_Clear );
	if( IsNull( message ) )
		return kPGPError_OutOfMemory;

	(*message)->messageID	= 0;
	(*message)->type		= kPGPldapType_None;
	(*message)->prev		= NULL;
	(*message)->next		= NULL;

	return err;
}


	static PGPError
pgpFreeLDAPMessageRef(
	PGPldapMessageRef	message,
	PGPBoolean			freeBER )
{
	PGPValidateLDAPMessageRef( message );

	/* Reset the pointers */
	if( message->next )
		(void) pgpFreeLDAPMessageRef( message->next, freeBER );

	if( message->prev )
		message->prev->next = NULL;

	if( freeBER )
		(void) PGPFreeBERElement( message->ber );

	(void) PGPFreeData( message );

	return kPGPError_NoErr;
}

	PGPError
PGPFreeLDAPContext(
	PGPldapContextRef	pgpLDAP )
{
	PGPValidateLDAPContextRef( pgpLDAP );

	if( PGPldapMessageRefIsValid( pgpLDAP->response ) )
		(void) pgpFreeLDAPMessageRef( pgpLDAP->response, TRUE );

	if( PGPSocketRefIsValid( pgpLDAP->sock ) )
		(void) PGPCloseSocket( pgpLDAP->sock );

	if( IsntNull (pgpLDAP->hostname ) )
		(void) PGPFreeData( pgpLDAP->hostname );

	if( IsntNull (pgpLDAP->matched ) )
		(void) PGPFreeData( pgpLDAP->matched );

	if( IsntNull (pgpLDAP->message ) )
		(void) PGPFreeData( pgpLDAP->message );

	(void) PGPFreeData( pgpLDAP );

	return kPGPError_NoErr;
}

/*
 * This frees not just the single PGPldapMessage, but all the
 * PGPldapMessages in the 'next' chain of PGPldapMessages.
 *
 * Note: even though each PGPldapMessage has a pointer to a PGPberElement,
 * as far as memory allocation is concerned, the PGPberElements actually
 * belong to the PGPldapContext and are freed when we call
 * PGPFreeLDAPContext, not PGPFreeLDAPMessage.
 */
	PGPError
PGPFreeLDAPMessage(
	PGPldapMessageRef	message )
{
	PGPValidateLDAPMessageRef( message );

	(void) pgpFreeLDAPMessageRef( message, FALSE );

	return kPGPError_NoErr;
}

	PGPError
PGPFreeLDAPValues(
	char **			vals )
{
	PGPUInt32		i	= 0;

	PGPValidatePtr( vals );

	for( i = 0; vals[i] != NULL; i++ )
		(void) PGPFreeData( vals[i] );

	(void) PGPFreeData( vals );

    return kPGPError_NoErr;
}

	PGPError
PGPFreeLDAPValuesLen(
	PGPberValue **	bvals )
{
	PGPUInt32		i	= 0;

	PGPValidatePtr( bvals );

	for( i = 0; bvals[i] != NULL; i++ )
	{
		(void) PGPFreeData( bvals[i]->value );
		(void) PGPFreeData( bvals[i] );
	}
	(void) PGPFreeData( bvals );

    return kPGPError_NoErr;
}

	PGPError
pgpNewLDAPURLDesc(
	PGPldapContextRef	pgpLDAP,
	PGPldapURLDesc **	lud )
{
	PGPError			err			= kPGPError_NoErr;

	PGPValidatePtr( lud );

	*lud = PGPNewData( pgpLDAP->memMgr, sizeof( PGPldapURLDesc ),
				kPGPMemoryMgrFlags_Clear );
	CKNULL( *lud );

	(*lud)->host	= NULL;
	(*lud)->port	= 0;
	(*lud)->dn		= NULL;
	(*lud)->attrs	= NULL;
	(*lud)->scope	= kPGPldapScope_Base;

	(*lud)->filter	= PGPNewData( pgpLDAP->memMgr,
						strlen( kPGPldap_DefaultFilterString ) + 1,
						kPGPMemoryMgrFlags_Clear );
	CKNULL( (*lud)->filter );

	pgpCopyMemory( kPGPldap_DefaultFilterString, (*lud)->filter,
		strlen( kPGPldap_DefaultFilterString ) + 1 );

	goto done;
error:

	if( IsntNull( *lud ) )
	{
		if( IsntNull( (*lud)->filter ) )
			(void) PGPFreeData( (*lud)->filter );
		(void) PGPFreeData( *lud );
	}

done:

	return err;
}

	PGPError
PGPFreeLDAPURLDesc(
	PGPldapURLDesc *	lud )
{
	PGPUInt32			i		= 0;

	PGPValidatePtr( lud );

	if( IsntNull( lud->host ) )
		(void) PGPFreeData( lud->host );

	if( IsntNull( lud->dn ) )
		(void) PGPFreeData( lud->dn );

	if( IsntNull( lud->attrs ) )
	{
		for( i = 0; lud->attrs[i] != NULL; i++ )
			(void) PGPFreeData( lud->attrs[i] );
		(void) PGPFreeData( lud->attrs );
	}

	if( IsntNull( lud->filter ) )
		(void) PGPFreeData( lud->filter );

	(void) PGPFreeData( lud );

	return kPGPError_NoErr;
}

	PGPError
PGPldapGetErrno(
	PGPldapContextRef	pgpLDAP,
	char **				matched,
	char **				message,
	PGPldapResult *		result )
{
	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidatePtr( result );

	*result = pgpLDAP->result;

	if( IsntNull( matched ) && ( matched[0] != '\0' ) )
		*matched = pgpLDAP->matched;

	if( IsntNull( message ) && ( message[0] != '\0' ) )
		*message = pgpLDAP->message;

	return kPGPError_NoErr;
}

	PGPError
PGPldapResultToError(
	PGPldapContextRef	pgpLDAP,
	PGPldapResult		result )
{
	PGPError					err					= kPGPError_NoErr;
	PGPUInt32					i					= 0;
	const PGPldapResultEntry *	entry				= NULL;

	(void) pgpLDAP;

	for( i = 0; i < kPGPldapResult_NumResultTableEntries; i++ )
	{
		entry = &sResults[i];

		if( entry->result == result )
		{
			err = entry->err;
			break;
		}
	}

	if( i == kPGPldapResult_NumResultTableEntries )
		err = kPGPError_LDAPOther;

	return err;
}

	static PGPError
pgpLDAPConnect(
	PGPldapContextRef	pgpLDAP,
	char *				host,
	PGPUInt16			port )
{
	PGPSocketAddressInternet	sin;
	PGPHostEntry				*hp 		= NULL;
	PGPInt32					i			= 0;
	PGPUInt32					ipAddress	= 0;
	PGPBoolean					bConnected	= FALSE;
	PGPError					err 		= kPGPError_NoErr;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidatePtr( host );

	err = PGPSocketsInit();
	if( IsPGPError( err ) )
	{
		err = PGPGetLastSocketsError();
		goto error;
	}

	pgpLDAP->sock = PGPOpenSocket(
		kPGPAddressFamilyInternet,
		kPGPSocketTypeStream,
		kPGPTCPProtocol );
	if( pgpLDAP->sock == kInvalidPGPSocketRef )
	{
		err = PGPGetLastSocketsError();
		goto error;
	}

	ipAddress = PGPDottedToInternetAddress( host );
	if( ipAddress != 0xFFFFFFFF )
	{
		(void) memset( &sin, 0, sizeof ( sin ) );
		sin.sin_family = kPGPAddressFamilyInternet;
		sin.sin_port = PGPHostToNetShort( port );
		(void) memcpy( &sin.sin_addr.s_addr, (void *) &ipAddress,
			sizeof( sin.sin_addr.s_addr ) );

		if( PGPConnect( pgpLDAP->sock,
				(PGPSocketAddress *) &sin,
				sizeof( sin ) ) >= 0 )
		{
			bConnected = TRUE;
		}
		else if( PGPGetLastSocketsError() == kPGPError_UserAbort )
		{
			err = kPGPError_UserAbort;
			goto error;
		}
	}

	if( !bConnected )
	{
		/* Connecting to the IP address didn't work */

		hp = PGPGetHostByName( host );
		if( IsNull ( hp ) )
			return PGPGetLastSocketsError();

		(void) memset( &sin, 0, sizeof ( sin ) );
		sin.sin_family = kPGPAddressFamilyInternet;
		sin.sin_port = PGPHostToNetShort( port );

		for( i = 0; hp->h_addr_list[i] != NULL; i++ )
		{
			(void) memcpy( &sin.sin_addr.s_addr, hp->h_addr_list[i],
				sizeof( sin.sin_addr.s_addr ) );
			if( PGPConnect( pgpLDAP->sock,
					(PGPSocketAddress *) &sin,
					sizeof( sin ) ) >= 0 )
			{
				bConnected = TRUE;
			}
			else if( PGPGetLastSocketsError() == kPGPError_UserAbort )
			{
				err = kPGPError_UserAbort;
				goto error;
			}
		}
	}

	if( bConnected )
	{
		/* Connected */
		pgpLDAP->hostname = PGPNewData( pgpLDAP->memMgr,
								strlen( host ) + 1,
								kPGPMemoryMgrFlags_Clear );
		if( IsNull( pgpLDAP->hostname ) )
		{
			err = kPGPError_OutOfMemory;
			goto error;
		}
		pgpCopyMemory( host, pgpLDAP->hostname, strlen( (char *)host ) + 1 );

		pgpLDAP->port = port;
	}
	else
	{
		if( !PGPSocketRefIsValid( pgpLDAP->sock ) )
			PGPCloseSocket( pgpLDAP->sock );

		/* Didn't connect */
		err = kPGPError_SocketsNotConnected;
	}

	goto done;
error:

done:
	return err;
}

	PGPError
pgpLDAPSend(
	PGPldapContextRef		pgpLDAP,
	PGPberElementRef		ber )
{
	PGPUInt32		bytesSent	= 0;
	PGPByte *		encoding	= NULL;
	PGPSize			length		= 0;
	PGPError		err			= kPGPError_NoErr;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidateBERElementRef( ber );

	if( pgpLDAP->sock == kInvalidPGPSocketRef )
	{
		err = kPGPError_SocketsNotConnected;
		goto error;
	}

	err = PGPberGetLength( ber, &length ); CKERR;
	err = PGPberGetEncoding( ber, &encoding ); CKERR;

	bytesSent = PGPSend( pgpLDAP->sock, encoding, length,
					kPGPSendFlagNone );

	if( bytesSent != length )
		err = PGPGetLastSocketsError();
error:
	return err;
}


	PGPError
PGPldapOpen(
	PGPldapContextRef	pgpLDAP,
	char *				host,
	PGPUInt16			port )
{
	PGPError			err 		= kPGPError_NoErr;
	char *				singleHost	= NULL;
	PGPBoolean			connected	= FALSE;
	PGPUInt32			i			= 0;
	PGPUInt32			j			= 0;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidatePtr( host );

	singleHost = PGPNewData( pgpLDAP->memMgr, strlen( host ) + 1,
					kPGPMemoryMgrFlags_Clear );
	CKNULL( singleHost );

	while( !connected )
	{
		while( host[i] == ' ' )
			i++;

		if( host[i] == '\0' )
		{
			pgpLDAP->result = kPGPldapResult_ConnectError;
			if( IsntPGPError( err ) )
				err = PGPldapResultToError( pgpLDAP, pgpLDAP->result );
			goto error;
		}

		j = 0;
		while( ( host[i] != ' ' ) && ( host[i] != '\0' ) )
		{
			singleHost[j++] = host[i++];
		}
		singleHost[j] = '\0';

		/* Connect to server */
		err = pgpLDAPConnect( pgpLDAP, singleHost, port );
		if( IsPGPError( err ) )
			continue;
		else
			connected = TRUE;
	}

	pgpLDAP->result = kPGPldapResult_Success;

	goto done;
error:

done:
	if( IsntNull( singleHost ) )
		(void) PGPFreeData( singleHost );

	return err;
}

	PGPError
PGPldapSimpleBindSync(
	PGPldapContextRef	pgpLDAP,
	char *				dn,
	char *				password )
{
	PGPError			err = kPGPError_NoErr;
	PGPldapMessageID	messageID = kInvalidPGPldapMessageID;
	PGPldapType 		messageType = kPGPldapType_None;
	PGPldapMessageRef	ldapMessage = kInvalidPGPldapMessageRef;

	PGPValidateLDAPContextRef( pgpLDAP );

	err = PGPldapSimpleBind (pgpLDAP, dn, password, &messageID ); CKERR;

	err = PGPNewLDAPMessage( pgpLDAP, &ldapMessage ); CKERR;

	err = PGPldapGetResult( pgpLDAP, messageID, FALSE, NULL, ldapMessage,
		&messageType ); CKERR;

error:
	if( PGPldapMessageRefIsValid( ldapMessage ) )
		(void) PGPFreeLDAPMessage( ldapMessage );

	return( err ? err : PGPldapResultToError( pgpLDAP, pgpLDAP->result ) );
}

	PGPError
PGPldapSimpleBind(
	PGPldapContextRef	pgpLDAP,
	char *				dn,
	char *				password,
	PGPldapMessageID *	messageID )
{
	PGPberElementRef 	ber 		= kInvalidPGPberElementRef;
	PGPError			err 		= kPGPError_NoErr;

	PGPValidatePtr( pgpLDAP );
	PGPValidatePtr( messageID );

	if( IsNull( dn ) )
		dn = (char *) "";
	if( IsNull( password ) )
		password = (char *) "";

	err = pgpNewBERElement( pgpLDAP->memMgr, &ber ); CKERR;

	err = PGPberAppend( ber, "{it{ists}}",
		++( pgpLDAP->nextMessageID ),
		kPGPldapRequest_Bind,
		kPGPldap_DefaultVersion,
		dn,
		kPGPldapAuth_Simple,
		password ); CKERR;

	err = pgpLDAPSend( pgpLDAP, ber ); CKERR;

	*messageID = pgpLDAP->nextMessageID;

error:

	if( PGPberElementRefIsValid( ber ) )
		(void) PGPFreeBERElement( ber );

	return err;
}

	PGPError
PGPldapBindSync(
	PGPldapContextRef	pgpLDAP,
	char *				dn,
	char *				password,
	PGPldapAuth 		method )
{
	PGPValidateLDAPContextRef( pgpLDAP );

	/* We only support simple authentication */
	if( method != kPGPldapAuth_Simple )
		return kPGPldapResult_StrongAuthNotSupported;

	return PGPldapSimpleBindSync( pgpLDAP, dn, password );
}

	PGPError
PGPldapBind(
	PGPldapContextRef	pgpLDAP,
	char *				dn,
	char *				password,
	PGPldapAuth 		method,
	PGPldapMessageID *	messageID )
{
	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidatePtr( messageID );

	/* We only support simple authentication */
	if( method != kPGPldapAuth_Simple )
		return kPGPldapResult_StrongAuthNotSupported;

	return PGPldapSimpleBind( pgpLDAP, dn, password, messageID );
}

	PGPError
PGPldapUnbind(
	PGPldapContextRef	pgpLDAP )
{
	PGPberElementRef	ber		= kInvalidPGPberElementRef;
	PGPError			err		= kPGPError_NoErr;

	PGPValidateLDAPContextRef( pgpLDAP );

	err = pgpNewBERElement( pgpLDAP->memMgr, &ber); CKERR;

	err = PGPberAppend(	ber, "{itn}",
			++(pgpLDAP->nextMessageID),
			kPGPldapRequest_Unbind ); CKERR;

	err = pgpLDAPSend( pgpLDAP, ber ); CKERR;

	if( PGPSocketRefIsValid( pgpLDAP->sock ) )
		(void) PGPCloseSocket( pgpLDAP->sock );
	pgpLDAP->sock = kInvalidPGPSocketRef;

	pgpLDAP->result = kPGPldapResult_Success;

	goto done;
error:
done:

	if( PGPberElementRefIsValid( ber ) )
		(void) PGPFreeBERElement( ber );

	return err;
}


	PGPError
PGPldapSetOption(
	PGPldapContextRef	pgpLDAP,
	PGPUInt32			option,
	void *				value )
{
	PGPValidateLDAPContextRef( pgpLDAP );

	switch( option )
	{
		case kPGPldapOpt_Deref:
			pgpLDAP->deref = (PGPldapDeref) value;
			break;

		case kPGPldapOpt_Sizelimit:
			pgpLDAP->sizelimit = (PGPUInt32) value;
			break;

		case kPGPldapOpt_Timelimit:
			pgpLDAP->timelimit = (PGPUInt32) value;
			break;

		case kPGPldapOpt_Desc:
			pgpLDAP->sock = (PGPSocketRef) value;
			break;

		default:
			return kPGPError_BadParams;

	}

	return kPGPError_NoErr;
}

	PGPError
PGPldapGetOption(
	PGPldapContextRef	pgpLDAP,
	PGPUInt32			option,
	void *				value )
{
	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidatePtr( value );

	switch( option )
	{
		case kPGPldapOpt_Deref:
			*((PGPldapDeref *) value) = pgpLDAP->deref;
			break;

		case kPGPldapOpt_Sizelimit:
			*((PGPUInt32 *) value) = pgpLDAP->sizelimit;
			break;

		case kPGPldapOpt_Timelimit:
			*((PGPUInt32 *) value) = pgpLDAP->timelimit;
			break;

		case kPGPldapOpt_Desc:
			*((PGPSocketRef *) value) = pgpLDAP->sock;
			break;

		default:
			return kPGPError_BadParams;

	}

	return kPGPError_NoErr;
}

	PGPError
pgpCopyLDAPMessage(
	PGPldapMessageRef	src,
	PGPldapMessageRef	dst )
{
	PGPValidateLDAPMessageRef( src );
	PGPValidateLDAPMessageRef( dst );

	dst->messageID	= src->messageID;
	dst->type		= src->type;

	/*
	 * Set this PGPldapMessageRef to point to the _same_
	 * PGPberElementRef -- it will get freed later when we
	 * free the PGPldapContextRef.
	 */
	dst->ber		= src->ber;

	return kPGPError_NoErr;
}

	static PGPError
pgpSeekLastLDAPMessage(
	PGPldapMessageRef	first,
	PGPldapMessageRef *	last )
{
	PGPValidateLDAPMessageRef( first );
	PGPValidatePtr( last );

	for( *last = first;
		 (*last)->next != kInvalidPGPldapMessageRef;
		 *last = (*last)->next )
		;	/* NULL */

	return kPGPError_NoErr;
}

	static PGPError
pgpAddLDAPMessageToLDAPContext(
	PGPldapContextRef	pgpLDAP,
	PGPldapMessageRef	message,
	PGPberElementRef	ber )
{
	PGPError			err			= kPGPError_NoErr;
	PGPldapMessageRef	lastMessage	= kInvalidPGPldapMessageRef;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidateLDAPMessageRef( message );
	PGPValidateBERElementRef( ber );

	message->ber = ber;

	err = PGPberRead( ber,
		"it",
		&(message->messageID),
		&(message->type) ); CKERR;

	/* Add message to pgpLDAP's list of responses */
	if( PGPldapMessageRefIsValid( pgpLDAP->response ) )
	{
		err = pgpSeekLastLDAPMessage( pgpLDAP->response, &lastMessage ); CKERR;

		lastMessage->next	= message;
		message->prev		= lastMessage;
	}
	else
		pgpLDAP->response = message;

error:
	return err;
}

	static PGPError
pgpReadLDAPResponse(
	PGPldapContextRef	pgpLDAP,
	PGPldapMessageRef *	message )
{
	PGPberElementRef	ber			= kInvalidPGPberElementRef;
	PGPldapMessageRef	newMessage	= kInvalidPGPldapMessageRef;
	PGPError			err			= kPGPError_NoErr;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidatePtr( message );

	err = pgpNewBERElement( pgpLDAP->memMgr, &ber ); CKERR;

	err = PGPberReadResponse( ber, pgpLDAP->sock ); CKERR;

	err = PGPNewLDAPMessage( pgpLDAP, &newMessage ); CKERR;

	err = pgpAddLDAPMessageToLDAPContext( pgpLDAP, newMessage, ber ); CKERR;

	*message = newMessage;

	goto done;

error:
	if( PGPberElementRefIsValid( ber ) )
		(void) PGPFreeBERElement( ber );
	ber = kInvalidPGPberElementRef;

	if( PGPldapMessageRefIsValid( newMessage ) )
		(void) PGPFreeLDAPMessage( newMessage );
	newMessage = kInvalidPGPldapMessageRef;

done:
	return err;
}

	PGPError
pgpLDAPGetMessageErrno(
	PGPldapContextRef		pgpLDAP,
	PGPldapMessageRef		message )
{
	PGPError				err		= kPGPError_NoErr;
	PGPberElementRef		ber		= kInvalidPGPberElementRef;
	PGPldapType				tag		= kPGPldapType_None;
	PGPSize					len		= 0;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidateLDAPMessageRef( message );

	ber = message->ber;

	if( IsntNull( pgpLDAP->matched ) )
		(void) PGPFreeData( pgpLDAP->matched );
	pgpLDAP->matched = NULL;

	if( IsntNull( pgpLDAP->message ) )
		(void) PGPFreeData( pgpLDAP->message );
	pgpLDAP->message = NULL;

	err = PGPberRewind( ber ); CKERR;

	err = PGPberRead( ber, "e", &( pgpLDAP->result ) ); CKERR;

	err = PGPberPeek( ber, (PGPberType *)&tag, &len ); CKERR;
	pgpLDAP->matched = PGPNewData( pgpLDAP->memMgr, len + 1,
							kPGPMemoryMgrFlags_Clear );
	if( IsNull( pgpLDAP->matched ) )
	{
		err = kPGPError_OutOfMemory;
		goto error;
	}
	err = PGPberRead( ber, "s", pgpLDAP->matched ); CKERR;

	err = PGPberPeek( ber, (PGPberType *)&tag, &len ); CKERR;
	pgpLDAP->message = PGPNewData( pgpLDAP->memMgr, len + 1,
							kPGPMemoryMgrFlags_Clear );
	if( IsNull( pgpLDAP->message ) )
	{
		err = kPGPError_OutOfMemory;
		goto error;
	}
	err = PGPberRead( ber, "s", pgpLDAP->message ); CKERR;

	goto done;
error:

	if( IsntNull( pgpLDAP->matched ) )
		(void) PGPFreeData( pgpLDAP->matched );
	pgpLDAP->matched = NULL;

	if( IsntNull( pgpLDAP->message ) )
		(void) PGPFreeData( pgpLDAP->message );
	pgpLDAP->message = NULL;

done:
	return err;
}


/*
 * This is the big LDAP function.  This is the function that gets
 * responses from an LDAP server and saves them away for later retrieval.
 *
 * If we are looking for the response of a specific PGPldapMessageID,
 * we need to: 1) look through our existing responses for it, or 2)
 * wait for it until we timeout.
 *
 * We also need to get any outstanding responses from the server and
 * save them away in the pgpLDAP->responses chain.
 */
	PGPError
PGPldapGetResult(
	PGPldapContextRef		pgpLDAP,
	PGPldapMessageID		messageID,
	PGPBoolean				all,
	PGPSocketsTimeValue *	timeout,
	PGPldapMessageRef		result,
	PGPldapType *			messageType )
{
	PGPldapMessageRef		message 		= kInvalidPGPldapMessageRef;
	PGPldapMessageRef		lastResult		= kInvalidPGPldapMessageRef;
	PGPldapMessageRef		prevLastResult	= kInvalidPGPldapMessageRef;
	PGPSocketSet			readSet;
	PGPSocketsTimeValue		tv				= { 0, 0 };
	PGPTime					startTime		= 0;
	PGPTime					endTime			= 0;
	PGPBoolean				found			= FALSE;
	PGPInt32				retval			= 0;
	PGPError				err				= kPGPError_NoErr;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidateLDAPMessageRef( result );
	PGPValidatePtr( messageType );

	lastResult = result;

	/*
	 * Look for any new responses we may have gotten and add them to
	 * the PGPldapContextRef
	 */

	PGPSOCKETSET_ZERO( &readSet );
	PGPSOCKETSET_SET( pgpLDAP->sock, &readSet );

	while( ((retval = PGPSelect( (PGPInt32) pgpLDAP->sock + 1, &readSet, NULL, NULL, &tv ))) )
	{
		if( retval < 0 )
		{
			err = PGPGetLastSocketsError();
			goto done;
		}
		err = pgpReadLDAPResponse( pgpLDAP, &message );	CKERR;
	}

	/* Look in our existing chain of server responses */
	for( message = pgpLDAP->response;
		 message != kInvalidPGPldapMessageRef;
		 message = message->next )
	{
		if( ( messageID == message->messageID ) ||
			( messageID == kPGPldapMessageID_Any ) )
		{
			/* Found it */
			if( !PGPldapMessageRefIsValid( lastResult ) )
			{
				/*
				 * We're returning a chain of PGPldapMessageRefs and
				 * we've already found one, so we need to make a new
				 * PGPldapMessageRef and add it to the end of our
				 * chain of responses.
				 */
				err = PGPNewLDAPMessage( pgpLDAP, &lastResult ); CKERR;

				prevLastResult->next	= lastResult;
				lastResult->prev		= prevLastResult;
			}

			err = pgpCopyLDAPMessage( message, lastResult ); CKERR;

			*messageType = message->type;

			if( !all )
			{
				found = TRUE;
				break;
			}

			if( ( messageID == message->messageID ) ||
				( messageID == kPGPldapMessageID_Any ) )
			{
				if( message->type == kPGPldapResponse_SearchResult )
				{
					found = TRUE;
					break;
				}
			}

			prevLastResult	= lastResult;
			lastResult		= kInvalidPGPldapMessageRef;
		}
	}

	if( !found )
	{
		/* Wait for the message (and save away any other messages we get) */
		if( IsntNull( timeout ) )
			tv = *timeout;

		/* NULL timeout means to block, tv = { 0, 0 } means check and quit */

		while( ( tv.tv_sec >= 0 ) || ( IsNull( timeout ) ) )
		{
			startTime = PGPGetTime();

			PGPSOCKETSET_ZERO( &readSet );
			PGPSOCKETSET_SET( pgpLDAP->sock, &readSet );

			if ( ((retval = PGPSelect( (PGPInt32) pgpLDAP->sock + 1, &readSet, NULL, NULL,
				( IsNull( timeout ) ? NULL : &tv ) ) ) ) )
			{
				/* Something came in or there was an error */
				if( retval < 0 )
				{
					err = PGPGetLastSocketsError();
					goto done;
				}

				err = pgpReadLDAPResponse( pgpLDAP, &message ); CKERR;

				if( ( messageID == message->messageID ) ||
					( messageID == kPGPldapMessageID_Any ) )
				{
					/* Found it */
					if( !PGPldapMessageRefIsValid( lastResult ) )
					{
						/* New PGPldapMessageRef at end of chain */
						err = PGPNewLDAPMessage( pgpLDAP, &lastResult ); CKERR;

						prevLastResult->next	= lastResult;
						lastResult->prev		= prevLastResult;
					}
				}

				err = pgpCopyLDAPMessage( message, lastResult ); CKERR;

				*messageType = message->type;

				if( !all )
				{
					found = TRUE;
					break;
				}

				if( ( messageID == message->messageID ) ||
					( messageID == kPGPldapMessageID_Any ) )
				{
					if( message->type == kPGPldapResponse_SearchResult )
					{
						found = TRUE;
						break;
					}
				}

				prevLastResult	= lastResult;
				lastResult		= kInvalidPGPldapMessageRef;
			}
			else	/* Timeout */
			{
				*messageType = kPGPldapType_None;
				break;
			}

			endTime = PGPGetTime();
			tv.tv_sec -= ( endTime - startTime );
		}
	}

	if( found )
	{
		if( *messageType != kPGPldapResponse_SearchEntry )
		{
			err = pgpLDAPGetMessageErrno( pgpLDAP, lastResult ); CKERR;
		}
		else
		{
			/*
			 * This is a pretty rare case, where we called PGPldapGetResult
			 * after doing a search, but didn't get the search result (yet).
			 * Since we don't have the LDAPResult, we can't fill in the
			 * PGPldapContext result fields.  Since there technically wasn't
			 * an error, we'll just say that everything went fine, but
			 * *messageType will indicate that we didn't find the search
			 * result.
			 */
			pgpLDAP->result = kPGPldapResult_Success;

			if( IsntNull( pgpLDAP->matched ) )
				(void) PGPFreeData( pgpLDAP->matched );
			pgpLDAP->matched = NULL;

			if( IsntNull( pgpLDAP->message ) )
				(void) PGPFreeData( pgpLDAP->message );
			pgpLDAP->message = NULL;
		}
	}
	else
	{
		/* Timeout */
		pgpLDAP->result = kPGPldapResult_Timeout;

		if( IsntNull( pgpLDAP->matched ) )
			(void) PGPFreeData( pgpLDAP->matched );
		pgpLDAP->matched = NULL;

		if( IsntNull( pgpLDAP->message ) )
			(void) PGPFreeData( pgpLDAP->message );
		pgpLDAP->message = NULL;
	}

	goto done;
error:

	*messageType = kPGPldapType_None;

done:
	return err;
}

	PGPError
PGPldapAbandon(
	PGPldapContextRef	pgpLDAP,
	PGPldapMessageID	messageID )
{
	PGPError			err			= kPGPError_NoErr;
	PGPberElementRef	ber			= kInvalidPGPberElementRef;

	err = pgpNewBERElement( pgpLDAP->memMgr, &ber ); CKERR;

	err = PGPberAppend( ber, "{iti}",
			++( pgpLDAP->nextMessageID ),
			kPGPldapRequest_Abandon,
			messageID ); CKERR;

	err = pgpLDAPSend( pgpLDAP, ber ); CKERR;

error:

	if( PGPberElementRefIsValid( ber ) )
		(void) PGPFreeBERElement( ber );

	return err;
}


	PGPError
PGPldapGetMessageID(
	PGPldapContextRef	pgpLDAP,
	PGPldapMessageRef	result,
	PGPldapMessageID *	messageID )
{
	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidateLDAPMessageRef( result );
	PGPValidatePtr( messageID );

	(void) pgpLDAP;

	*messageID = result->messageID;

	return kPGPError_NoErr;
}

	PGPError
PGPldapSearchSync(
	PGPldapContextRef	pgpLDAP,
	char *				base,
	PGPldapScope		scope,
	char *				filter,
	char *				attrs[],
	PGPBoolean			attrsOnly,
	PGPldapMessageRef	resultMessage )
{
	return PGPldapSearchSyncTimeout( pgpLDAP, base, scope, filter, attrs,
				attrsOnly, NULL, resultMessage );
}

	PGPError
PGPldapSearchSyncTimeout(
	PGPldapContextRef		pgpLDAP,
	char *					base,
	PGPldapScope			scope,
	char *					filter,
	char *					attrs[],
	PGPBoolean				attrsOnly,
	PGPSocketsTimeValue *	tv,
	PGPldapMessageRef		resultMessage )
{
	PGPError			err			= kPGPError_NoErr;
	PGPldapMessageID	messageID	= kInvalidPGPldapMessageID;
	PGPldapType			messageType	= kPGPldapType_None;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidatePtr( filter );
	PGPValidateLDAPMessageRef( resultMessage );

	err = PGPldapSearch( pgpLDAP, base, scope, filter, attrs, attrsOnly,
				&messageID ); CKERR;

	err = PGPldapGetResult( pgpLDAP, messageID, TRUE, tv, resultMessage,
				&messageType ); CKERR;

error:
	return( err ? err : PGPldapResultToError( pgpLDAP, pgpLDAP->result ) );
}

	static PGPError
pgpLDAPGetFilterType(
	char *					filter,
	PGPldapFilter *			filterType )
{
	PGPUInt32		i	= 0;
	PGPUInt32		j	= 0;
	PGPError		err = kPGPError_NoErr;

	PGPValidatePtr( filter );
	PGPValidatePtr( filterType );

	/* filter[0] should be '(' */
	if( filter[0] != '(' )
	{
		err = kPGPError_BadParams;
		goto error;
	}

	switch( filter[1] )
	{
		case '&':
			*filterType = kPGPldapFilter_And;
			break;

		case '|':
			*filterType = kPGPldapFilter_Or;
			break;

		case '!':
			*filterType = kPGPldapFilter_Not;
			break;

		default:
			for( i = 0;
				 ( filter[i] != '=' ) && ( filter[i] != '<' ) &&
				 ( filter[i] != '>' ) && ( filter[i] != '~' ) &&
				 ( filter[i] != '\0');
				 i++ )
				; /* NULL */

			if( filter[i] == '\0' )
			{
				err = kPGPError_BadParams;
				goto error;
			}

			switch( filter[i] )
			{
				case '=':
					if( ( filter[i+1] == '*' ) && ( filter[i+2] == ')' ) )
					{
						*filterType = kPGPldapFilter_Present;
						break;	/* out of the switch(...) */
					}

					for( j = i + 1;
						 ( filter[j] != ')' ) &&
						 ( filter[j] != '\0');
						 j++ )
					{
						if( filter[j] == '*' )
						{
							*filterType = kPGPldapFilter_Substrings;
							break;	/* out of the for(...) */
						}
					}
					if( *filterType != kPGPldapFilter_Substrings )
						*filterType = kPGPldapFilter_Equal;
					break;

				case '<':
					if( filter[i+1] != '=' )
					{
						err = kPGPError_BadParams;
						goto error;
					}

					*filterType = kPGPldapFilter_LE;
					break;

				case '>':
					if( filter[i+1] != '=' )
					{
						err = kPGPError_BadParams;
						goto error;
					}

					*filterType = kPGPldapFilter_GE;
					break;

				case '~':
					if( filter[i+1] != '=' )
					{
						err = kPGPError_BadParams;
						goto error;
					}

					*filterType = kPGPldapFilter_Approx;
					break;

				default:
					err = kPGPError_BadParams;
					goto error;
			}
	}

error:

	return err;
}

	PGPError
pgpLDAPGetSubstring(
	char *					value,
	char *					substring,
	PGPldapSubstring *		substringType,
	PGPSize *				inc )
{
	PGPBoolean				start		= FALSE;
	PGPBoolean				end			= FALSE;
	PGPUInt32				i			= 0;
	PGPUInt32				j			= 0;
	PGPError				err			= kPGPError_NoErr;

	PGPValidatePtr( value );
	PGPValidatePtr( substring );
	PGPValidatePtr( substringType );
	PGPValidatePtr( inc );

	if( value[0] == '*' )
	{
		start = TRUE;
		value++;
	}

	for( i = 0, j = 0; ( value[j] != '*' ) && ( value[j] != '\0' ); i++, j++ )
	{
		if( value[j] == '\\' )
			j++;
		substring[i] = value[j];
	}
	substring[i] = '\0';
	*inc = j;

	if( value[j] == '*' )
		end = TRUE;

	/*
	 * If we didn't find any *'s in the value, we should assume that
	 * we've already found them all in previous calls to
	 * pgpLDAPGetSubstrings, and this is just the end of value, as in:
	 *
	 *	"*string1*string2"
	 *
	 * In which case the first call would have found
	 *	"string1", kPGPldapSubstring_Any,
	 * and the second call would have found
	 *	"string2", kPGPldapSubstring_Final
	 */
	if( start && end )
		*substringType = kPGPldapSubstring_Any;
	else if( end )
		*substringType = kPGPldapSubstring_Initial;
	else
		*substringType = kPGPldapSubstring_Final;

	return err;
}

	PGPError
pgpLDAPGetSubfilter(
	char *					filter,
	char *					subfilter )
{
	PGPUInt32				i		= 0;
	PGPUInt32				j		= 0;
	PGPUInt32				parens	= 0;

	PGPValidatePtr( filter );
	PGPValidatePtr( subfilter );

	/*
	 * filter is of the form (...)(...)(...)...
	 * All we really need to do is copy everything in the first set of (...)
	 * to subfilter (including the parens).
	 */

	for( i = 0, j = 0; filter[i] != '\0'; i++, j++ )
	{
		subfilter[j] = filter[i];

		if( filter[i] == '(' )
			parens++;
		if( filter[i] == ')' )
			parens--;

		if( filter[i] == ')' && parens == 0 )
			break;
	}

	subfilter[j]	= ')';
	subfilter[j+1]	= '\0';

	return kPGPError_NoErr;
}

	PGPError
pgpLDAPAddFilter(
	PGPldapContextRef		pgpLDAP,
	PGPberElementRef		ber,
	char *					inFilter )
{
	PGPError				err				= kPGPError_NoErr;
	PGPUInt32				i				= 0;
	PGPUInt32				j				= 0;
	char *					subfilter		= NULL;
	char *					filter			= NULL;
	PGPldapFilter			filterType		= kPGPldapFilter_None;
	char *					attr			= NULL;
	char *					value			= NULL;
	char *					substring		= NULL;
	PGPSize					length			= 0;
	PGPSize					offset			= 0;
	PGPSize					valueIndex		= 0;
	PGPSize					valueInc		= 0;
	PGPldapSubstring		substringType	= kPGPldapSubstring_None;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidateBERElementRef( ber );
	PGPValidatePtr( inFilter );

	/*
	 * A Filter looks like this:
	 *	Filter ::=
	 *		CHOICE {
	 *			and            [0] SET OF Filter,
	 *			or             [1] SET OF Filter,
	 *			not            [2] Filter,
	 *			equalityMatch  [3] AttributeValueAssertion,
	 *			substrings     [4] SubstringFilter,
	 *			greaterOrEqual [5] AttributeValueAssertion,
	 *			lessOrEqual    [6] AttributeValueAssertion,
	 *			present        [7] AttributeType,
	 *			approxMatch    [8] AttributeValueAssertion
	 *		}
	 * Note: tags in a choice are always explicit
	 */

	length = strlen( (char *)inFilter );
	length++;

	if( ( inFilter[0] != '(' ) && ( inFilter[length-2] !=')' ) )
		length += 2;

	filter = PGPNewData( pgpLDAP->memMgr, length, kPGPMemoryMgrFlags_Clear );
	if( IsNull( filter ) )
		goto error;

	if( ( inFilter[0] != '(' ) && ( inFilter[length-2] !=')' ) )
	{
		filter[0] = '(';
		pgpCopyMemory( inFilter, filter + 1, length - 3 );
		filter[length-2] = ')';
		filter[length-1] = '\0';
	}
	else
		pgpCopyMemory( inFilter, filter, length );

	attr = PGPNewData( pgpLDAP->memMgr, length, kPGPMemoryMgrFlags_Clear );
	if( IsNull( attr ) )
		goto error;

	substring = PGPNewData( pgpLDAP->memMgr, length, kPGPMemoryMgrFlags_Clear );
	if( IsNull( substring ) )
		goto error;

	subfilter = PGPNewData( pgpLDAP->memMgr, length, kPGPMemoryMgrFlags_Clear );
	if( IsNull( subfilter ) )
		goto error;

	value = PGPNewData( pgpLDAP->memMgr, length, kPGPMemoryMgrFlags_Clear );
	if( IsNull( value ) )
		goto error;

	err = pgpLDAPGetFilterType( filter, &filterType );
	if( IsPGPError( err ) )
		goto error;

	switch( filterType )
	{
		case kPGPldapFilter_And:
		case kPGPldapFilter_Or:

			/*
			 * Both kPGPldapFilter_And and kPGPldapFilter_Or append
			 * a "SET OF Filter" to the ber
			 */

			/* filter[length-1] is NULL, filter[length-2] is ')' */

			filter[length - 2] = '\0';
			offset = 2;			/* skip the "(&" or "(|" */

			err = PGPberAppend( ber, "t[", filterType ); CKERR;

			while( offset < ( length - 2 ) )
			{
				err = pgpLDAPGetSubfilter( filter + offset, subfilter ); CKERR;
				offset += strlen( (char *)subfilter );

				err = pgpLDAPAddFilter( pgpLDAP, ber, subfilter ); CKERR;
			}

			err = PGPberAppend( ber, "]" ); CKERR;

			break;

		case kPGPldapFilter_Not:

			err = PGPberAppend( ber, "t{", filterType ); CKERR;

			filter[length - 2] = '\0';

			err = pgpLDAPAddFilter( pgpLDAP, ber, filter + 2 ); CKERR;

			err = PGPberAppend( ber, "}" ); CKERR;

			break;

		case kPGPldapFilter_Equal:
		case kPGPldapFilter_GE:
		case kPGPldapFilter_LE:
		case kPGPldapFilter_Approx:

			/* Append an "AttributeTypeAssertion" to the ber */

			for( i = 1, j = 0;
				 ( filter[i] != '=' ) && ( filter[i] != '<' ) &&
				 ( filter[i] != '>' ) && ( filter[i] != '~' );
				 i++, j++ )
			{
				attr[j] = filter[i];
			}
			attr[j] = '\0';

			if( filter[i] != '=' )
				i++;

			for( i++, j = 0; filter[i] != ')'; i++, j++ )
				value[j] = filter[i];
			value[j] = '\0';

			err = PGPberAppend( ber, "t{ss}",
				filterType,
				attr,
				value ); CKERR;

			break;

		case kPGPldapFilter_Substrings:
			/* filter: "(attr=<value>)" */
			for( i = 1; filter[i] != '='; i++ )
				attr[i-1] = filter[i];
			attr[i-1] = '\0';

			/* i points to the '=' */
			i++;

			for( j = i; filter[j] != ')'; j++ )
				value[j-i] = filter[j];
			value[j-i] = '\0';

			err = PGPberAppend( ber, "t{s{",
					kPGPldapFilter_Substrings,
					attr ); CKERR;

			while( valueIndex < strlen( (char *)value ) )
			{
				if( ( value[valueIndex] == '*' ) && ( value[valueIndex+1] == '\0' ) )
					break;

				err = pgpLDAPGetSubstring( value + valueIndex, substring,
						&substringType, &valueInc ); CKERR;

				if( value[valueIndex] == '*' )
					valueIndex++;
				valueIndex += valueInc;

				if( substring[0] == '\0' )
				{
					/* Empty string - don't do anything */
					/* valueIndex++; */
					continue;
				}

				err = PGPberAppend( ber, "ts",
						substringType,
						substring ); CKERR;
			}

			err = PGPberAppend( ber, "}}" ); CKERR;

			break;

		case kPGPldapFilter_Present:

			for( i = 1, j = 0; filter[i] != '='; i++, j++ )
				attr[j] = filter[i];
			attr[j] = '\0';

			err = PGPberAppend( ber, "ts",
					filterType,
					attr ); CKERR;

			break;

		default:
			break;
	}

	goto done;
error:

done:
	if( IsntNull( filter ) )
		(void) PGPFreeData( filter );

	if( IsntNull( attr ) )
		(void) PGPFreeData( attr );

	if( IsntNull( substring ) )
		(void) PGPFreeData( substring );

	if( IsntNull( subfilter ) )
		(void) PGPFreeData( subfilter );

	if( IsntNull( value ) )
		(void) PGPFreeData( value );

	return err;
}

	PGPError
PGPldapSearch(
	PGPldapContextRef	pgpLDAP,
	char *				base,
	PGPldapScope		scope,
	char *				filter,
	char *				attrs[],
	PGPBoolean			attrsOnly,
	PGPldapMessageID *	messageID )
{
	PGPError			err		= kPGPError_NoErr;
	PGPberElementRef	ber		= kInvalidPGPberElementRef;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidatePtr( filter );
	PGPValidatePtr( messageID );

	if( IsNull( base ) )
		base = (char *) "";

	err = pgpNewBERElement( pgpLDAP->memMgr, &ber ); CKERR;

	/*
	 * Create the search request.  It looks like this:
	 *	SearchRequest ::=
	 *		[APPLICATION 3] SEQUENCE {
	 *			 baseObject     LDAPDN,
	 *			 scope          ENUMERATED {
	 *								 baseObject            (0),
	 *								 singleLevel           (1),
	 *								 wholeSubtree          (2)
	 *							},
	 *			 derefAliases   ENUMERATED {
	 *								 neverDerefAliases     (0),
	 *								 derefInSearching      (1),
	 *								 derefFindingBaseObj   (2),
	 *								 alwaysDerefAliases    (3)
	 *							},
	 *			 sizeLimit      INTEGER (0 .. maxInt),
	 *							-- value of 0 implies no sizelimit
	 *			 timeLimit      INTEGER (0 .. maxInt),
	 *							-- value of 0 implies no timelimit
	 *			 attrsOnly     BOOLEAN,
	 *							-- TRUE, if only attributes (without values)
	 *							-- to be returned.
	 *			 filter         Filter,
	 *			 attributes     SEQUENCE OF AttributeType
	 *		}
	 * wrapped in an ldap message.
	 */

	err = PGPberAppend( ber, "{it{seeiib",
			++(pgpLDAP->nextMessageID),
			kPGPldapRequest_Search,
			base,
			scope,
			pgpLDAP->deref,
			pgpLDAP->sizelimit,
			pgpLDAP->timelimit,
			attrsOnly ); CKERR;

	err = pgpLDAPAddFilter( pgpLDAP, ber, filter ); CKERR;

	err = PGPberAppend( ber, "{v}}}", attrs ); CKERR;

	err = pgpLDAPSend( pgpLDAP, ber ); CKERR;

	*messageID = pgpLDAP->nextMessageID;

	goto done;
error:

	*messageID = kInvalidPGPldapMessageID;

done:

	if( PGPberElementRefIsValid( ber ) )
		(void) PGPFreeBERElement( ber );

	return err;
}

	PGPError
PGPldapCountEntries(
	PGPldapContextRef	pgpLDAP,
	PGPldapMessageRef	result,
	PGPInt32 *			num )
{
	PGPldapMessageRef		message		= kInvalidPGPldapMessageRef;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidateLDAPMessageRef( result );
	PGPValidatePtr( num );

	(void) pgpLDAP;

	*num = 0;

	for( message = result;
		 message != kInvalidPGPldapMessageRef;
		 message = message->next )
	{
		if( message->type == kPGPldapResponse_SearchEntry )
			(*num)++;
	}

	return kPGPError_NoErr;
}

	PGPError
PGPldapFirstEntry(
	PGPldapContextRef	pgpLDAP,
	PGPldapMessageRef	result,
	PGPldapMessageRef *	outMessage )
{
	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidateLDAPMessageRef( result );
	PGPValidatePtr( outMessage );

	*outMessage = result;

	return kPGPError_NoErr;
}

	PGPError
PGPldapNextEntry(
	PGPldapContextRef	pgpLDAP,
	PGPldapMessageRef	prevEntry,
	PGPldapMessageRef *	outMessage )
{
	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidateLDAPMessageRef( prevEntry );
	PGPValidatePtr( outMessage );

	*outMessage = prevEntry->next;

	if( IsNull( *outMessage ) )
	{
		*outMessage = kInvalidPGPldapMessageRef;
		goto done;
	}

	if( (*outMessage)->type == kPGPldapResponse_SearchResult )
		*outMessage = kInvalidPGPldapMessageRef;

done:
	return kPGPError_NoErr;
}

	PGPError
PGPldapGetDN(
	PGPldapContextRef	pgpLDAP,
	PGPldapMessageRef	entry,
	char **				dn )
{
	PGPError			err			= kPGPError_NoErr;
	PGPberElementRef	ber			= kInvalidPGPberElementRef;
	PGPberType			tag			= kPGPberType_None;
	PGPSize				len			= 0;
	PGPUInt32			index		= 0;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidateLDAPMessageRef( entry );
	PGPValidatePtr( dn );

	if( entry->type != kPGPldapResponse_SearchEntry )
		return kPGPError_BadParams;

	ber = entry->ber;

	err = PGPberGetIndex( ber, &index ); CKERR;

	err = PGPberRewind( ber ); CKERR;

	/* Skip MessageID */
	err = PGPberNextPrimitive( ber ); CKERR;
	err = PGPberSkip( ber ); CKERR;

	err = PGPberNextPrimitive( ber ); CKERR;
	err = PGPberPeek( ber, &tag, &len ); CKERR;

	*dn = PGPNewData( pgpLDAP->memMgr, len + 1, kPGPMemoryMgrFlags_Clear );
	if( IsNull( *dn ) )
	{
		err = kPGPError_OutOfMemory;
		goto error;
	}

	err = PGPberRead( ber, "s", *dn ); CKERR;

	err = PGPberSetIndex( ber, index ); CKERR;

error:

	return err;
}

	PGPError
PGPldapFirstAttribute(
	PGPldapContextRef	pgpLDAP,
	PGPldapMessageRef	entry,
	PGPberElementRef *	ber,
	char **				attr )
{
	PGPError			err			= kPGPError_NoErr;
	PGPldapType			tag			= kPGPldapType_None;
	PGPSize				len			= 0;
	PGPUInt32			index		= 0;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidateLDAPMessageRef( entry );
	PGPValidatePtr( ber );
	PGPValidatePtr( attr );

	*ber = entry->ber;

	err = PGPberRewind( *ber ); CKERR;

	/* Skip MessageID */
	err = PGPberNextPrimitive( *ber ); CKERR;
	err = PGPberSkip( *ber ); CKERR;

	/* Skip DN */
	err = PGPberNextPrimitive( *ber ); CKERR;
	err = PGPberSkip( *ber ); CKERR;

	err = PGPberNextConstructed( *ber );
	if( err == kPGPError_EndOfIteration )
	{
		/*
		 * There doesn't appear to be anything else.  This could be 
		 * something funny like a referral.  At any rate, it's not an
		 * error, so we should handle it gracefully.
		 */

		 *attr = NULL;
		 err = kPGPError_NoErr;
		 goto done;
	}
	CKERR;
	if( (*ber)->nextReadIndex == (*ber)->length )
	{
		*attr = NULL;
		goto done;
	}

	err = PGPberGetLength( *ber, &len ); CKERR;
	err = PGPberGetIndex( *ber, &index ); CKERR;
	if( len == index )
	{
		/* No attributes */
		*attr = NULL;
		goto done;
	}

	/* Read attribute */
	err = PGPberNextPrimitive( *ber ); CKERR;

	err = PGPberPeek( *ber, (PGPberType *)&tag, &len ); CKERR;
	*attr = PGPNewData( (*ber)->memMgr, len + 1, kPGPMemoryMgrFlags_Clear );
	if( IsNull( *attr ) )
	{
		err = kPGPError_OutOfMemory;
		goto error;
	}

	err = PGPberRead( *ber, "s", *attr ); CKERR;

error:
done:
	return err;
}

	PGPError
PGPldapNextAttribute(
	PGPldapContextRef	pgpLDAP,
	PGPldapMessageRef	entry,
	PGPberElementRef	ber,
	char **				attr )
{
	PGPError			err			= kPGPError_NoErr;
	PGPldapType			tag			= kPGPldapType_None;
	PGPSize				len			= 0;
	PGPSize				berLength	= 0;
	PGPUInt32			index		= 0;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidateLDAPMessageRef( entry );
	PGPValidateBERElementRef( ber );
	PGPValidatePtr( attr );

	/*
	 * The only way to get a valid PGPberElementRef is via
	 * PGPldapFirstAttribute, so we don't really need to have a more
	 * elaborate method of determining if we've already called
	 * PGPldapFirstAttribute.
	 */

	/* We've already called PGPldapFirstAttribute, so we can skip the values */
	err = PGPberSkip( ber ); CKERR;

	err = PGPberGetIndex( ber, &index ); CKERR;
	err = PGPberGetLength( ber, &berLength ); CKERR;
	if( index == berLength )
	{
		*attr = NULL;
		goto done;
	}

	err = PGPberPeek( ber, (PGPberType *)&tag, &len ); CKERR;

	*attr = PGPNewData( pgpLDAP->memMgr, len + 1, kPGPMemoryMgrFlags_Clear );
	if( IsNull( *attr ) )
	{
		err = kPGPError_OutOfMemory;
		goto error;
	}

	err = PGPberRead( ber, "s", *attr ); CKERR;

error:
done:
	return err;
}

	PGPError
pgpLDAPCountValues(
	PGPldapContextRef	pgpLDAP,
	PGPberElementRef	ber,
	PGPSize	*			num )
{
	PGPError			err			= kPGPError_NoErr;
	PGPUInt32			startIndex	= 0;
	PGPUInt32			index		= 0;
	PGPSize				setLength	= 0;
	PGPSize				len			= 0;
	PGPldapType			tag			= kPGPldapType_None;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidateBERElementRef( ber );
	PGPValidatePtr( num );

	*num = 0;

	err = PGPberGetIndex( ber, &startIndex ); CKERR;

	err = PGPberPeek( ber, (PGPberType *)&tag, &setLength ); CKERR;
	err = PGPberGetIndex( ber, &index ); CKERR;
	while( index < startIndex + setLength )
	{
		err = PGPberNextPrimitive( ber ); CKERR;
		err = PGPberPeek( ber, (PGPberType *)&tag, &len ); CKERR;

		err = PGPberSkip( ber ); CKERR;

		(*num)++;

		err = PGPberGetIndex( ber, &index ); CKERR;
	}

	err = PGPberSetIndex( ber, startIndex ); CKERR;

error:
	return err;
}

	PGPError
PGPldapGetValues(
	PGPldapContextRef	pgpLDAP,
	PGPldapMessageRef	entry,
	char *				attr,
	char ***			values )
{
	PGPError			err			= kPGPError_NoErr;
	PGPUInt32			i			= 0;
	PGPSize				numValues	= 0;
	PGPberValue **		berValues	= NULL;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidateLDAPMessageRef( entry );
	PGPValidatePtr( attr );
	PGPValidatePtr( values );


	err = PGPldapGetValuesLen( pgpLDAP, entry, attr, &berValues ); CKERR;

	if( IsNull( berValues ) )
	{
		err = kPGPError_LDAPNoSuchAttribute;
		goto error;
	}

	err = PGPldapCountValuesLen( pgpLDAP, berValues, &numValues ); CKERR;
	
	(*values) = PGPNewData( pgpLDAP->memMgr,
					( numValues + 1 ) * sizeof( char * ),
					kPGPMemoryMgrFlags_Clear );
	if( IsNull( *values ) )
	{
		err = kPGPError_OutOfMemory;
		goto error;
	}

	for( i = 0; IsntNull( berValues[i] ); i++ )
	{
		(*values)[i] = PGPNewData( pgpLDAP->memMgr,
							berValues[i]->length + 1,
							kPGPMemoryMgrFlags_Clear );
		if( IsNull( (*values)[i] ) )
		{
			err = kPGPError_OutOfMemory;
			goto error;
		}

		pgpCopyMemory(
			berValues[i]->value,
			(*values)[i],
			berValues[i]->length );
		(*values)[i][berValues[i]->length] = '\0';
	}
	(*values)[i] = NULL;

	goto done;
error:

	if( IsntNull( *values ) )
		(void) PGPFreeLDAPValues( *values );
	*values = NULL;

done:

	if( IsntNull( berValues ) )
		(void) PGPFreeLDAPValuesLen( berValues );

	return err;
}

	PGPError
PGPldapGetValuesLen(
	PGPldapContextRef	pgpLDAP,
	PGPldapMessageRef	entry,
	char *				attr,
	PGPberValue ***		values )
{
	PGPberElementRef	ber			= kInvalidPGPberElementRef;
	PGPldapType			tag			= kPGPldapType_None;
	char *				attribute	= NULL;
	PGPSize				numValues	= 0;
	PGPSize				len			= 0;
	PGPUInt32			startIndex	= 0;
	PGPUInt32			i			= 0;
	PGPError			err			= kPGPError_NoErr;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidateLDAPMessageRef( entry );
	PGPValidatePtr( attr );
	PGPValidatePtr( values );

	ber = entry->ber;

	err = PGPberGetIndex( ber, &startIndex ); CKERR;

	err = PGPldapFirstAttribute( pgpLDAP, entry, &ber, &attribute ); CKERR;
	while( IsntNull( attribute ) )
	{
		if( pgpCompareStringsIgnoreCase( (char *)attribute, (char *)attr ) == 0 )
		{
			err = pgpLDAPCountValues( pgpLDAP, ber, &numValues ); CKERR;

			*values = PGPNewData( pgpLDAP->memMgr,
						( numValues + 1 ) * sizeof( PGPberValue * ),
						kPGPMemoryMgrFlags_Clear );
			if( IsNull( *values ) )
			{
				err = kPGPError_OutOfMemory;
				goto error;
			}

			/* Skip the set tag - go right to the string */
			err = PGPberNextPrimitive( ber ); CKERR;

			for( i = 0; i < numValues; i++ )
			{
				err = PGPberPeek( ber, (PGPberType *)&tag, &len ); CKERR;
				(*values)[i] = PGPNewData( pgpLDAP->memMgr,
									sizeof( PGPberValue ),
									kPGPMemoryMgrFlags_Clear );
				if( IsNull( (*values)[i] ) )
				{
					err = kPGPError_OutOfMemory;
					goto error;
				}
				(*values)[i]->value = PGPNewData( pgpLDAP->memMgr, len + 1,
									kPGPMemoryMgrFlags_Clear );
				if( IsNull( (*values)[i] ) )
				{
					err = kPGPError_OutOfMemory;
					goto error;
				}

				(*values)[i]->length = len;

				err = PGPberRead( ber, "o", (*values)[i]->value ); CKERR;
			}
			(*values)[i] = NULL;

			goto done;
		}

		(void) PGPFreeData( attribute );
		err = PGPldapNextAttribute( pgpLDAP, entry, ber, &attribute ); CKERR;
	}
	/* Not found */
	*values = NULL;
	err = kPGPError_LDAPNoSuchAttribute;

error:
done:
	(void) PGPberSetIndex( ber, startIndex );

	if( IsntNull( attribute ) )
		(void) PGPFreeData( attribute );

	return err;
}

	PGPError
PGPldapCountValues(
	PGPldapContextRef	pgpLDAP,
	char **				values,
	PGPSize *			num )
{
	PGPSize				i		= 0;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidatePtr( values );
	PGPValidatePtr( num );

	for( i = 0; IsntNull( values[i] ); i++ )
		; /* NULL */

	*num = i;

	return kPGPError_NoErr;
}


	PGPError
PGPldapCountValuesLen(
	PGPldapContextRef	pgpLDAP,
	PGPberValue **		values,
	PGPSize *			num )
{
	PGPSize				i		= 0;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidatePtr( values );
	PGPValidatePtr( num );

	for( i = 0; IsntNull( values[i] ); i++ )
		; /* NULL */

	*num = i;

	return kPGPError_NoErr;
}

	PGPError
PGPldapModify(
	PGPldapContextRef	pgpLDAP,
	char *				dn,
	PGPldapMod *		mod[],
	PGPldapMessageID *	messageID )
{
	PGPError			err		= kPGPError_NoErr;
	PGPberElementRef	ber		= kInvalidPGPberElementRef;
	PGPUInt32			i		= 0;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidatePtr( mod );
	PGPValidatePtr( messageID );

	if( IsNull( dn ) )
		dn = (char *) "";

	err = pgpNewBERElement( pgpLDAP->memMgr, &ber ); CKERR;

	err = PGPberAppend( ber, "{it{s{",
			++( pgpLDAP->nextMessageID ),
			kPGPldapRequest_Modify,
			dn ); CKERR;

	for( i = 0; mod[i] != NULL; i++ )
	{
		err = PGPberAppend( ber, "{e",
				mod[i]->op & ~kPGPldapModOpMask_UseBERValues ); CKERR;

		if( mod[i]->op & kPGPldapModOpMask_UseBERValues )
		{
			err = PGPberAppend( ber, "{s[V]}}",
				mod[i]->type,
				mod[i]->bvalue ); CKERR;
		}
		else
		{
			err = PGPberAppend( ber, "{s[v]}}",
				mod[i]->type,
				mod[i]->value ); CKERR;
		}
	}

	err = PGPberAppend( ber, "}}}" ); CKERR;

	err = pgpLDAPSend( pgpLDAP, ber ); CKERR;

	*messageID = pgpLDAP->nextMessageID;

error:

	if( PGPberElementRefIsValid( ber ) )
		(void) PGPFreeBERElement( ber );

	return err;
}

	PGPError
PGPldapModifySync(
	PGPldapContextRef	pgpLDAP,
	char *				dn,
	PGPldapMod *		mod[] )
{
	PGPError			err				= kPGPError_NoErr;
	PGPldapMessageID	messageID		= kInvalidPGPldapMessageID;
	PGPldapType			messageType		= kPGPldapType_None;
	PGPldapMessageRef	resultMessage	= kInvalidPGPldapMessageRef;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidatePtr( mod );

	err = PGPldapModify( pgpLDAP, dn, mod, &messageID ); CKERR;

	err = PGPNewLDAPMessage( pgpLDAP, &resultMessage ); CKERR;

	err = PGPldapGetResult( pgpLDAP, messageID, FALSE, NULL, resultMessage,
				&messageType ); CKERR;

error:
	if( PGPldapMessageRefIsValid( resultMessage ) )
		(void) PGPFreeLDAPMessage( resultMessage );

	return( err ? err : PGPldapResultToError( pgpLDAP, pgpLDAP->result ) );
}

	PGPError
PGPldapAdd(
	PGPldapContextRef	pgpLDAP,
	char *				dn,
	PGPldapMod *		mod[],
	PGPldapMessageID *	messageID )
{
	PGPError			err		= kPGPError_NoErr;
	PGPberElementRef	ber		= kInvalidPGPberElementRef;
	PGPUInt32			i		= 0;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidatePtr( mod );
	PGPValidatePtr( messageID );

	if( IsNull( dn ) )
		dn = (char *) "";

	err = pgpNewBERElement( pgpLDAP->memMgr, &ber ); CKERR;

	err = PGPberAppend( ber, "{it{s{",
			++( pgpLDAP->nextMessageID ),
			kPGPldapRequest_Add,
			dn ); CKERR;

	for( i = 0; mod[i] != NULL; i++ )
	{
		if( mod[i]->op & kPGPldapModOpMask_UseBERValues )
		{
			err = PGPberAppend( ber, "{s[V]}",
				mod[i]->type,
				mod[i]->bvalue ); CKERR;
		}
		else
		{
			err = PGPberAppend( ber, "{s[v]}",
				mod[i]->type,
				mod[i]->value ); CKERR;
		}
	}

	err = PGPberAppend( ber, "}}}" ); CKERR;

	err = pgpLDAPSend( pgpLDAP, ber ); CKERR;

	*messageID = pgpLDAP->nextMessageID;

error:

	if( PGPberElementRefIsValid( ber ) )
		(void) PGPFreeBERElement( ber );

	return err;
}

	PGPError
PGPldapAddSync(
	PGPldapContextRef	pgpLDAP,
	char *				dn,
	PGPldapMod *		mod[] )
{
	PGPError			err				= kPGPError_NoErr;
	PGPldapMessageID	messageID		= kInvalidPGPldapMessageID;
	PGPldapType			messageType		= kPGPldapType_None;
	PGPldapMessageRef	resultMessage	= kInvalidPGPldapMessageRef;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidatePtr( mod );

	err = PGPldapAdd( pgpLDAP, dn, mod, &messageID ); CKERR;

	err = PGPNewLDAPMessage( pgpLDAP, &resultMessage ); CKERR;

	err = PGPldapGetResult( pgpLDAP, messageID, FALSE, NULL, resultMessage,
				&messageType ); CKERR;

error:
	if( PGPldapMessageRefIsValid( resultMessage ) )
		(void) PGPFreeLDAPMessage( resultMessage );

	return( err ? err : PGPldapResultToError( pgpLDAP, pgpLDAP->result ) );
}

	PGPError
PGPldapDelete(
	PGPldapContextRef	pgpLDAP,
	char *				dn,
	PGPldapMessageID *	messageID )
{
	PGPError			err		= kPGPError_NoErr;
	PGPberElementRef	ber		= kInvalidPGPberElementRef;

	PGPValidatePtr( messageID );
	PGPValidatePtr( dn );

	err = pgpNewBERElement( pgpLDAP->memMgr, &ber ); CKERR;

	err = PGPberAppend( ber, "{its}",
			++( pgpLDAP->nextMessageID ),
			kPGPldapRequest_Delete,
			dn ); CKERR;

	err = pgpLDAPSend( pgpLDAP, ber ); CKERR;

	*messageID = pgpLDAP->nextMessageID;

error:

	if( PGPberElementRefIsValid( ber ) )
		(void) PGPFreeBERElement( ber );

	return err;
}

	PGPError
PGPldapDeleteSync(
	PGPldapContextRef	pgpLDAP,
	char *				dn )
{
	PGPError			err				= kPGPError_NoErr;
	PGPldapMessageID	messageID		= kInvalidPGPldapMessageID;
	PGPldapType			messageType		= kPGPldapType_None;
	PGPldapMessageRef	resultMessage	= kInvalidPGPldapMessageRef;

	PGPValidateLDAPContextRef( pgpLDAP );

	err = PGPldapDelete( pgpLDAP, dn, &messageID ); CKERR;

	err = PGPNewLDAPMessage( pgpLDAP, &resultMessage ); CKERR;

	err = PGPldapGetResult( pgpLDAP, messageID, FALSE, NULL, resultMessage,
				&messageType ); CKERR;

error:
	if( PGPldapMessageRefIsValid( resultMessage ) )
		(void) PGPFreeLDAPMessage( resultMessage );

	return( err ? err : PGPldapResultToError( pgpLDAP, pgpLDAP->result ) );
}

	PGPError
PGPldapModifyRDN(
	PGPldapContextRef	pgpLDAP,
	char *				dn,
	char *				newRDN,
	PGPBoolean			deleteOldRDN,
	PGPldapMessageID *	messageID )
{
	PGPError			err				= kPGPError_NoErr;
	PGPberElementRef	ber				= kInvalidPGPberElementRef;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidatePtr( dn );
	PGPValidatePtr( newRDN );
	PGPValidatePtr( messageID );

	err = pgpNewBERElement( pgpLDAP->memMgr, &ber ); CKERR

	err = PGPberAppend( ber, "{it{ssb}}",
			++( pgpLDAP->nextMessageID ),
			kPGPldapRequest_ModifyRDN,
			dn,
			newRDN,
			deleteOldRDN ); CKERR;

	err = pgpLDAPSend( pgpLDAP, ber ); CKERR;

	*messageID = pgpLDAP->nextMessageID;

error:

	if( PGPberElementRefIsValid( ber ) )
		(void) PGPFreeBERElement( ber );

	return err;
}

	PGPError
PGPldapModifyRDNSync(
	PGPldapContextRef	pgpLDAP,
	char *				dn,
	char *				newRDN,
	PGPBoolean			deleteOldRDN )
{
	PGPError			err				= kPGPError_NoErr;
	PGPldapMessageID	messageID		= kInvalidPGPldapMessageID;
	PGPldapType			messageType		= kPGPldapType_None;
	PGPldapMessageRef	resultMessage	= kInvalidPGPldapMessageRef;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidatePtr( dn );
	PGPValidatePtr( newRDN );

	err = PGPldapModifyRDN( pgpLDAP, dn, newRDN, deleteOldRDN, &messageID ); CKERR;

	err = PGPNewLDAPMessage( pgpLDAP, &resultMessage ); CKERR;

	err = PGPldapGetResult( pgpLDAP, messageID, FALSE, NULL, resultMessage,
				&messageType ); CKERR;

error:

	if( PGPldapMessageRefIsValid( resultMessage ) )
		(void) PGPFreeLDAPMessage( resultMessage );

	return( err ? err : PGPldapResultToError( pgpLDAP, pgpLDAP->result ) );
}

	PGPError
PGPldapCompare(
	PGPldapContextRef	pgpLDAP,
	char *				dn,
	char *				type,
	char *				value,
	PGPldapMessageID *	messageID )
{
	PGPError			err			= kPGPError_NoErr;
	PGPberElementRef	ber			= kInvalidPGPberElementRef;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidatePtr( dn );
	PGPValidatePtr( type );
	PGPValidatePtr( value );
	PGPValidatePtr( messageID );

	err = pgpNewBERElement( pgpLDAP->memMgr, &ber ); CKERR;

	err = PGPberAppend( ber, "{it{s{ss}}}",
			++( pgpLDAP->nextMessageID ),
			kPGPldapRequest_Compare,
			dn,
			type,
			value ); CKERR;

	err = pgpLDAPSend( pgpLDAP, ber ); CKERR;

	*messageID = pgpLDAP->nextMessageID;

error:

	if( PGPberElementRefIsValid( ber ) )
		(void) PGPFreeBERElement( ber );

	return err;
}

	PGPError
PGPldapCompareSync(
	PGPldapContextRef	pgpLDAP,
	char *				dn,
	char *				type,
	char *				value,
	PGPBoolean *		equal )
{
	PGPError			err				= kPGPError_NoErr;
	PGPldapMessageID	messageID		= kInvalidPGPldapMessageID;
	PGPldapType			messageType		= kPGPldapType_None;
	PGPldapMessageRef	resultMessage	= kInvalidPGPldapMessageRef;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidateLDAPContextRef( pgpLDAP );

	err = PGPldapCompare( pgpLDAP, dn, type, value, &messageID ); CKERR;

	err = PGPNewLDAPMessage( pgpLDAP, &resultMessage ); CKERR;

	err = PGPldapGetResult( pgpLDAP, messageID, FALSE, NULL, resultMessage,
				&messageType ); CKERR;

	err = PGPldapGetCompareResult( pgpLDAP, messageID, equal ); CKERR;

error:
	if( PGPldapMessageRefIsValid( resultMessage ) )
		(void) PGPFreeLDAPMessage( resultMessage );

	return( err ? err : PGPldapResultToError( pgpLDAP, pgpLDAP->result ) );
}

	PGPError
PGPldapGetCompareResult(
	PGPldapContextRef	pgpLDAP,
	PGPldapMessageID	messageID,
	PGPBoolean *		equal )
{
	PGPError			err				= kPGPError_NoErr;
	PGPldapType			messageType		= kPGPldapType_None;
	PGPldapMessageRef	resultMessage	= kInvalidPGPldapMessageRef;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidatePtr( equal );

	err = PGPNewLDAPMessage( pgpLDAP, &resultMessage ); CKERR;

	err = PGPldapGetResult( pgpLDAP, messageID, FALSE, NULL, resultMessage,
				&messageType ); CKERR;

	err = PGPFreeLDAPMessage( resultMessage ); CKERR;

	if( pgpLDAP->result == kPGPldapResult_CompareTrue )
		*equal = TRUE;
	else if( pgpLDAP->result == kPGPldapResult_CompareFalse )
		*equal = FALSE;
	else
	{
		err = PGPldapResultToError( pgpLDAP, pgpLDAP->result );
		goto error;
	}

error:

	return err;
}

	PGPError
PGPldapIsLDAPURL(
	PGPldapContextRef	pgpLDAP,
	char *				url,
	PGPBoolean *		isURL )
{
	PGPError			err			= kPGPError_NoErr;
	PGPSize				len			= 0;
	char *				s			= NULL;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidatePtr( url );
	PGPValidatePtr( isURL );

	len = strlen( (char *)url );

	s = url;

	if( url[0] == '<' )
	{
		if( url[len-1] != '>' )
		{
			*isURL = FALSE;
			goto done;
		}
		s++;
	}

	if( pgpCompareStringsIgnoreCaseN( s, "URL:", 4 ) == 0 )
		s += 4;

	if( ( pgpCompareStringsIgnoreCaseN( s, "ldap://",  7 ) == 0 ) ||
		( pgpCompareStringsIgnoreCaseN( s, "ldaps://", 8 ) == 0 ) )
		*isURL = TRUE;
	else
		*isURL = FALSE;

done:
	return err;
}

	PGPError
PGPldapURLParse(
	PGPldapContextRef	pgpLDAP,
	char *				url,
	PGPldapURLDesc **	outLDAPDesc )
{
	PGPError			err			= kPGPError_NoErr;
	PGPUInt32			len			= 0;
	char *				temp		= NULL;
	char *				s			= NULL;
	char				sPort[16];
	PGPldapURLDesc *	lud			= NULL;
	PGPUInt32			i			= 0;
	PGPUInt32			j			= 0;
	PGPUInt32			numAttrs	= 0;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidatePtr( url );
	PGPValidatePtr( outLDAPDesc );

	/*
	 * LDAP URLs look like this:
	 *	ldap://host:port/dn[?attributes[?scope[?filter]]]
	 * They can also look like:
	 *	<ldapurl> and <URL:ldapurl>
	 */

	len = strlen( url );

	if( url[0] == '<' )
	{
		if( url[len-1] == '>' )
		{
			len -= 2;

			temp = PGPNewData( pgpLDAP->memMgr, len + 1, kPGPMemoryMgrFlags_Clear );
			CKNULL( temp );

			pgpCopyMemory( url + 1, temp, len );
			temp[len] = '\0';
		}
		else
		{
			err = kPGPError_BadParams;
			goto error;
		}
	}
	else
	{
		temp = PGPNewData( pgpLDAP->memMgr, len + 1, kPGPMemoryMgrFlags_Clear );
		CKNULL( temp );

		pgpCopyMemory( url, temp, len );
		temp[len] = '\0';
	}

	s = temp;

	if( pgpCompareStringsIgnoreCaseN( s, "URL:", 4 ) == 0 )
		s += 4;

	if( pgpCompareStringsIgnoreCaseN( s, "ldap://", 7 ) == 0 )
		s += 7;
	else if( pgpCompareStringsIgnoreCaseN( s, "ldaps://", 8 ) == 0)
		s += 8;
	else
	{
		err = kPGPError_LDAPNotLDAPURL;
		goto error;
	}

	err = pgpNewLDAPURLDesc( pgpLDAP, &lud ); CKERR;

	*outLDAPDesc = lud;

	if( *s != '/' )
	{
		/* There is a host in this url */
		for( i = 0; ( s[i] != ':' ) && ( s[i] != '/' ) && ( s[i] != '\0' ); i++ )
			; /* NULL */

		lud->host = PGPNewData( pgpLDAP->memMgr, i + 1, kPGPMemoryMgrFlags_Clear );
		CKNULL( lud->host );

		for( i = 0; ( s[i] != ':' ) && ( s[i] != '/' ) && ( s[i] != '\0' ); i++ )
			lud->host[i] = s[i];
		lud->host[i] = '\0';

		/* Skip the host */
		s += i;
	}

	if( s[0] == ':' )
	{
		/* There's a port */
		/* Skip the ':' */
		s++;

		for( i = 0; ( s[i] != '/' ) && ( s[i] != '\0' ); i++ )
			sPort[i] = s[i];
		sPort[i] = '\0';

		/* Skip the port */
		s += i;

		lud->port = atoi( sPort );
	}

	if( s[0] != '/' )
	{
		err = kPGPError_LDAPNoDN;
		goto error;
	}
	else
		s++;

	for( i = 0; ( s[i] != '?' ) && ( s[i] != '\0' ); i++ )
		;	/* NULL */

	lud->dn = PGPNewData( pgpLDAP->memMgr, i + 1, kPGPMemoryMgrFlags_Clear );
	CKNULL( lud->dn );

	for( i = 0; ( s[i] != '?' ) && ( s[i] != '\0' ); i++ )
		lud->dn[i] = s[i];
	lud->dn[i] = '\0';

	/* Skip the DN */
	s += i;

	if( s[0] == '\0' )
		goto done;

	/* Skip the '?' */
	s++;

	/* Count the attributes */
	numAttrs = 0;
	for( i = 0; ( s[i] != '?' ) && ( s[i] != '\0' ); i++ )
	{
		if( s[i] != ',' )
			numAttrs++;

		while( ( s[i] != ',' ) && ( s[i+1] != '?' ) && ( s[i+1] != '\0' ) )
			i++;
	}

	lud->attrs = PGPNewData( pgpLDAP->memMgr,
					( numAttrs + 1 ) * sizeof( char * ),
					kPGPMemoryMgrFlags_Clear );
	CKNULL( lud->attrs );

	for( j = 0; j < numAttrs; j++ )
	{
		/* Eat any extra whitespace */
		while( *s == ' ' )
			s++;

		for( i = 0; ( s[i] != ',' ) && ( s[i] != '?' ) && ( s[i] != '\0' ); i++ )
			;	/* NULL */

		lud->attrs[j] = PGPNewData( pgpLDAP->memMgr, i + 1, kPGPMemoryMgrFlags_Clear );
		CKNULL( lud->attrs[j] );

		for( i = 0; ( s[i] != ',' ) && ( s[i] != '?' ) && ( s[i] != '\0' ); i++ )
			lud->attrs[j][i] = s[i];
		lud->attrs[j][i] = '\0';

		if( s[i] == ',' )
			i++;

		s += i;
	}
	lud->attrs[j] = NULL;

	if( s[0] == '\0' )
		goto done;

	/* Skip the '?' */
	s++;

	for( i = 0; ( s[i] != '?' ) && ( s[i] != '\0' ); i++ )
		; /* NULL */

	if( i >= strlen( kPGPldapScopeString_Base ) )
	{
		if( pgpCompareStringsIgnoreCaseN( s, kPGPldapScopeString_Base,
				strlen( kPGPldapScopeString_Base ) ) == 0 )
			lud->scope = kPGPldapScope_Base;
	}
	if( i >= strlen( kPGPldapScopeString_OneLevel ) )
	{
		if( pgpCompareStringsIgnoreCaseN( s, kPGPldapScopeString_OneLevel,
				strlen( kPGPldapScopeString_OneLevel ) ) == 0 )
			lud->scope = kPGPldapScope_OneLevel;
	}
	if( i >= strlen( kPGPldapScopeString_Subtree ) )
	{
		if( pgpCompareStringsIgnoreCaseN( s, kPGPldapScopeString_Subtree,
				strlen( kPGPldapScopeString_Subtree ) ) == 0 )
			lud->scope = kPGPldapScope_Subtree;
	}

	/* Skip the scope */
	s += i;

	if( s[0] == '\0' )
		goto done;

	/* Skip the '?' */
	s++;

	for( i = 0; s[i] != '\0'; i++ )
		;	/* NULL */

	/*
	 * The default value for lud->filter is the string "(objectclass=*)",
	 * so lud->filter needs to be freed before we re-allocate it.
	 */
	if( IsntNull( lud->filter ) )
		(void) PGPFreeData( lud->filter );

	lud->filter = PGPNewData( pgpLDAP->memMgr, i + 1, kPGPMemoryMgrFlags_Clear );
	CKNULL( lud->filter );

	pgpCopyMemory( s, lud->filter, i + 1 );

	goto done;
error:

	if( IsntNull( lud ) )
		(void) PGPFreeLDAPURLDesc( lud );
	*outLDAPDesc = NULL;

done:

	if( IsntNull( temp ) )
		(void) PGPFreeData( temp );

	return err;
}

	PGPError
PGPldapURLSearch(
	PGPldapContextRef	pgpLDAP,
	char *				url,
	PGPBoolean			attrsOnly,
	PGPldapMessageID *	messageID )
{
	PGPError			err			= kPGPError_NoErr;
	PGPldapURLDesc *	desc		= NULL;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidatePtr( url );
	PGPValidatePtr( messageID );

	err = PGPldapURLParse( pgpLDAP, url, &desc ); CKERR;

	/*
	 * This is bizarre.  One would think that you could just pass a complete
	 * URL to this function and it would get the host, port, everything
	 * from the URL and start the search.  Instead, we require that we have
	 * already connected and bound to an LDAP server before we can
	 * use the URL.  To ensure this, we assert that the URL does not specify
	 * a host or a port.  This is for compatibility with the UMich LDAP SDK.
	 */
	if( IsntNull( desc->host ) || ( desc->port != 0 ) )
	{
		err = kPGPError_BadParams;
		goto error;
	}

	err = PGPldapSearch( pgpLDAP, desc->dn, desc->scope, desc->filter,
			desc->attrs, attrsOnly,	messageID ); CKERR;

error:

	if( IsntNull( desc ) )
		(void) PGPFreeLDAPURLDesc( desc );

	return err;
}

	PGPError
PGPldapURLSearchSync(
	PGPldapContextRef	pgpLDAP,
	char *				url,
	PGPBoolean			attrsOnly,
	PGPldapMessageRef	resultMessage )
{
	return PGPldapURLSearchSyncTimeout( pgpLDAP, url, attrsOnly,
				NULL, resultMessage );
}

	PGPError
PGPldapURLSearchSyncTimeout(
	PGPldapContextRef		pgpLDAP,
	char *					url,
	PGPBoolean				attrsOnly,
	PGPSocketsTimeValue *	tv,
	PGPldapMessageRef		resultMessage )
{
	PGPError			err			= kPGPError_NoErr;
	PGPldapMessageID	messageID	= kInvalidPGPldapMessageID;
	PGPldapType			messageType	= kPGPldapType_None;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidateLDAPMessageRef( resultMessage );
	PGPValidatePtr( url );

	err = PGPldapURLSearch( pgpLDAP, url, attrsOnly, &messageID ); CKERR;

	err = PGPldapGetResult( pgpLDAP, messageID, TRUE, tv, resultMessage,
				&messageType ); CKERR;

error:

	return( err ? err : PGPldapResultToError( pgpLDAP, pgpLDAP->result ) );
}

	PGPError
PGPldapExplodeDN(
	PGPldapContextRef	pgpLDAP,
	char *				dn,
	PGPBoolean			noTypes,
	char ***			components )
{
	PGPError			err			= kPGPError_NoErr;
	PGPSize				num			= 0;
	char *				s			= NULL;
	PGPUInt32			i			= 0;
	PGPUInt32			j			= 0;
	char **				attrs		= NULL;

	s = dn;

	/* Count the attributes */
	num = 0;
	for( i = 0; ( s[i] != '?' ) && ( s[i] != '\0' ); i++ )
	{
		if( s[i] != ',' )
			num++;

		while( ( s[i] != ',' ) && ( s[i+1] != '?' ) && ( s[i+1] != '\0' ) )
			i++;
	}

	attrs = PGPNewData( pgpLDAP->memMgr,
					( num + 1 ) * sizeof( char * ),
					kPGPMemoryMgrFlags_Clear );
	CKNULL( attrs );

	for( j = 0; j < num; j++ )
	{
		/* Eat any extra whitespace */
		while( *s == ' ' )
			s++;

		if( noTypes )
		{
			for( i = 0; ( s[i] != ',' ) && ( s[i] != '=' ) && ( s[i] != '\0' ); i++ )
				; /* NULL */

			s += i + 1;
		}

		for( i = 0; ( s[i] != ',' ) && ( s[i] != '\0' ); i++ )
			;	/* NULL */

		attrs[j] = PGPNewData( pgpLDAP->memMgr, i + 1, kPGPMemoryMgrFlags_Clear );
		CKNULL( attrs[j] );

		for( i = 0; ( s[i] != ',' ) && ( s[i] != '\0' ); i++ )
			attrs[j][i] = s[i];
		attrs[j][i] = '\0';

		if( s[i] == ',' )
			i++;

		s += i;
	}
	attrs[j] = NULL;

	*components = attrs;

	goto done;
error:

	if( IsntNull( attrs ) )
		(void) PGPFreeLDAPValues( attrs );

done:
	return err;
}

	PGPInt32
pgpLDAPSortCompare(
	const void *		s1,
	const void *		s2 )
{
	PGPError			err				= kPGPError_NoErr;
	PGPldapMessageSort	sort1;
	PGPldapMessageSort	sort2;
	char **				values1			= NULL;
	char **				values2			= NULL;
	char *				dn1				= NULL;
	char *				dn2				= NULL;
	PGPInt32			compareResult	= 0;
	PGPUInt32			i				= 0;

	pgpAssert( s1 != NULL );
	pgpAssert( s2 != NULL );

	pgpCopyMemory( s1, &sort1, sizeof( PGPldapMessageSort ) );
	pgpCopyMemory( s2, &sort2, sizeof( PGPldapMessageSort ) );

	if( IsNull( sort1.attr ) )
	{
		/* Use DN */
		err = PGPldapGetDN( sort1.pgpLDAP, sort1.message, &dn1 ); CKERR;
		err = PGPldapGetDN( sort2.pgpLDAP, sort2.message, &dn2 ); CKERR;

		compareResult = sort1.compare( &dn1, &dn2 );
	}
	else
	{
		/* Use attr */
		err = PGPldapGetValues( sort1.pgpLDAP, sort1.message, sort1.attr, &values1 ); CKERR;
		err = PGPldapGetValues( sort2.pgpLDAP, sort2.message, sort2.attr, &values2 ); CKERR;

		for( i = 0;	( values1[i] != NULL ) && ( values2[i] != NULL ); i++ )
		{
			compareResult = sort1.compare( &values1[i], &values2[i] );
			if( compareResult != 0 )
				break;
		}
	}

error:

	if( IsntNull( dn1 ) )
		(void) PGPFreeData( dn1 );
	if( IsntNull( dn2 ) )
		(void) PGPFreeData( dn2 );

	if( IsntNull( values1 ) )
		(void) PGPFreeLDAPValues( values1 );
	if( IsntNull( values2 ) )
		(void) PGPFreeLDAPValues( values2 );

	return compareResult;
}

	PGPError
PGPldapSortEntries(
	PGPldapContextRef	pgpLDAP,
	PGPldapMessageRef	chain,
	char *				attr,
	PGPInt32			(*compare)( char ** a, char ** b ),
	PGPldapMessageRef	sortedChain )
{
	PGPError				err			= kPGPError_NoErr;
	PGPldapMessageSort *	sorted		= NULL;
	PGPSize					num			= 0;
	PGPSize					i			= 0;
	PGPldapMessageRef		temp		= kInvalidPGPldapMessageRef;
	PGPldapMessageRef		prev		= kInvalidPGPldapMessageRef;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidateLDAPMessageRef( chain );
	PGPValidateLDAPMessageRef( sortedChain );
	PGPValidatePtr( compare );

	/*
	 * Copy the PGPldapMessageRefs into an array of struct PGPldapMessageSorts
	 */
	err = PGPldapCountEntries( pgpLDAP, chain, (PGPInt32 *) &num ); CKERR;

	sorted = PGPNewData( pgpLDAP->memMgr, num * sizeof( PGPldapMessageSort ),
				kPGPMemoryMgrFlags_Clear );
	CKNULL( sorted );

	err = PGPldapFirstEntry( pgpLDAP, chain, &temp ); CKERR;
	for( i = 0; i < num; i++ )
	{
		sorted[i].message = temp;
		sorted[i].pgpLDAP = pgpLDAP;
		sorted[i].attr = attr;
		sorted[i].compare = (void *) compare;

		err = PGPldapNextEntry( pgpLDAP, temp, &temp ); CKERR;
	}

	/* Sort the array */
	qsort( (void *) sorted, num, sizeof( PGPldapMessageSort ),
		pgpLDAPSortCompare );

	/*
	 * Copy the array of struct PGPldapMessageSorts back into a chain of
	 * PGPldapMessageRefs
	 */
	temp = sortedChain;
	for( i = 0; i < num; i++ )
	{
		if( !PGPldapMessageRefIsValid( temp ) )
		{
			err = PGPNewLDAPMessage( pgpLDAP, &temp );
			CKERR;
		}

		err = pgpCopyLDAPMessage( sorted[i].message, temp ); CKERR;

		temp->prev = prev;
		temp->next = kInvalidPGPldapMessageRef;

		if( PGPldapMessageRefIsValid( prev ) )
			prev->next = temp;

		prev = temp;

		temp = kInvalidPGPldapMessageRef;
	}

	goto done;
error:

	temp = sortedChain->next;
	while( PGPldapMessageRefIsValid( temp ) )
	{
		prev = temp;
		temp = prev->next;
		(void) PGPFreeLDAPMessage( prev );
	}

done:

	if( IsntNull( sorted ) )
		(void) PGPFreeData( sorted );

	return err;
}

	PGPError
PGPldapSortValues(
	PGPldapContextRef	pgpLDAP,
	char **				vals,
	PGPInt32			(*compare)( char ** a, char ** b ),
	char ***			sortedVals )
{
	PGPError			err			= kPGPError_NoErr;
	PGPUInt32			num			= 0;
	PGPUInt32			i			= 0;
	PGPSize				len			= 0;

	PGPValidateLDAPContextRef( pgpLDAP );
	PGPValidatePtr( vals );
	PGPValidatePtr( sortedVals );
	PGPValidatePtr( compare );

	for( num = 0; IsntNull( vals[num] ); num++ )
		;	/* NULL */

	*sortedVals = PGPNewData( pgpLDAP->memMgr, ( num + 1 ) * sizeof( char * ),
					kPGPMemoryMgrFlags_Clear );
	CKNULL( *sortedVals );

	for( i = 0; i < num; i++ )
	{
		len = strlen( (char *)vals[i] ) + 1;

		(*sortedVals)[i] = PGPNewData( pgpLDAP->memMgr, len,
							kPGPMemoryMgrFlags_Clear );
		CKNULL( (*sortedVals)[i] );

		pgpCopyMemory( vals[i], (*sortedVals)[i], len );
	}
	(*sortedVals)[i] = NULL;

	qsort( *sortedVals, num, sizeof( char * ), (void *) compare );

	goto done;
error:

	if( IsntNull( *sortedVals ) )
		(void) PGPFreeLDAPValues( *sortedVals );

done:

	return err;
}

	PGPInt32
PGPldapCompareStringsIgnoreCase(
	char **			s1,
	char **			s2 )
{
	return pgpCompareStringsIgnoreCase( *s1, *s2 );
}

/*__Editor_settings____

	Local Variables:
	tab-width: 4
	End:
	vi: ts=4 sw=4
	vim: si
_____________________*/
